package gov.va.med.mhv.vitals.validator;

import java.text.MessageFormat;
import java.util.Date;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.vitals.dto.BloodPressureReadingDTO;
import gov.va.med.mhv.vitals.util.MessagesUtil;

@Component
public class BloodPressureValidator extends Validator{
	
	private static final Integer DIASTOLIC_MIN_VALUE = new Integer("1");
	private static final Integer DIASTOLIC_MAX_VALUE = new Integer("999");
	public static final int COMMENTS_MAX_LENGTH = 255;
	private static final Integer SYSTOLIC_MIN_VALUE = new Integer("10");
	private static final Integer SYSTOLIC_MAX_VALUE = new Integer("999");
	public static final String SYSTOLIC_PROPERTY = "Systolic ";
	public static final String DIASTOLIC_PROPERTY = "Diastolic ";

	
	@Autowired
	private MessagesUtil messages;
	
	public void bpReadingValidations(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		fieldValidation(response, bpdto);
		if (response.getValidationErrors().size() == 0) {
			complexValidation(response, bpdto);
		}
	}

	
	protected void fieldValidation(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		validateDiastolic(response,bpdto);
		validateComments(response,bpdto);
		validateDateEntered(response,bpdto);
		validateSystolic(response,bpdto);
		validateUserProfile(response,bpdto);
	}	
	
	protected void complexValidation(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		validateSystolicDiastolic(response,bpdto);
	}
	
	private void validateSystolicDiastolic(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		
		if( bpdto.getDiastolic() != null && bpdto.getSystolic() != null ) {
			if( bpdto.getDiastolic().compareTo( bpdto.getSystolic()  ) >= 0 ) {
				response.getValidationErrors().put("sei.bloodpressure.detail.error.invalidComparison.systolic",
						messages.getInvalidSystolicCompare());
				response.getValidationErrors().put("sei.bloodpressure.detail.error.invalidComparison.diastolic",
						messages.getInvalidDiastolicCompare());
			}
		}
	}
	
	protected void validateDiastolic(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		Integer value = bpdto.getDiastolic();
		if( value == null ) {
			Validations.validateRequired(DIASTOLIC_PROPERTY, value, response.getValidationErrors());
			return;
		}		
		Validations.validateMinValue("diastolicminimum", value, DIASTOLIC_MIN_VALUE, DIASTOLIC_PROPERTY,response.getValidationErrors());
		Validations.validateMaxValue("diastolicmaximum",value, DIASTOLIC_MAX_VALUE,DIASTOLIC_PROPERTY,response.getValidationErrors());
	}	
	
	
	protected void validateComments(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		String value = bpdto.getComments();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("comments",value, COMMENTS_MAX_LENGTH, response.getValidationErrors());
	}	
	
	protected void validateDateEntered(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		Date value = bpdto.getReading();
		if( value == null ) {
			Validations.validateRequired("dateEntered", value, response.getValidationErrors());
			return;
		}		
	}	
	
	
	protected void validateSystolic(ResponseUtil response, BloodPressureReadingDTO bpdto) {
		Integer value = bpdto.getSystolic();
		if( value == null ) {
			Validations.validateRequired(SYSTOLIC_PROPERTY, value, response.getValidationErrors());
			return;
		}		
		
		Validations.validateMinValue("systolicminimum", value, SYSTOLIC_MIN_VALUE,SYSTOLIC_PROPERTY, response.getValidationErrors());
		Validations.validateMaxValue("systolicmaximum",value, SYSTOLIC_MAX_VALUE,SYSTOLIC_PROPERTY,response.getValidationErrors());
	}	
	
	protected void validateUserProfile(ResponseUtil response, BloodPressureReadingDTO bpdto) {

		if( bpdto.getUserprofileId() == null ) {
			response.getValidationErrors().put("userProfile", "userProfile is required");
			return;
		}
		
		boolean valid= isUserProfileValid(bpdto.getUserprofileId());
		if ( !valid) {
			response.getValidationErrors().put("userProfile", "userProfile is Invalid");
		}
	}
	
	public void addError(ResponseUtil response, String errorKey,String value, Object[]  args) {
		response.getValidationErrors().put(errorKey,MessageFormat.format(value, args));
	}  

}
